define(['angular', 'jQuery'], function (angular, $) {
	"use strict";

	angular.module('CalendarDatePicker', ['ui.bootstrap.datepicker', 'ui.bootstrap.dateparser', 'ui.bootstrap.position'])

		.constant('datepickerConfig', {
			formatDay: 'dd',
			formatMonth: 'MMMM',
			formatYear: 'yyyy',
			formatDayHeader: 'EEE',
			formatDayTitle: 'MMMM yyyy',
			formatMonthTitle: 'yyyy',
			datepickerMode: 'day',
			minMode: 'day',
			maxMode: 'year',
			showWeeks: true,
			startingDay: 0,
			yearRange: 20,
			minDate: null,
			maxDate: null
		})

		.controller('DatepickerController', function($scope, $attrs, $parse, $interpolate, $timeout, $log, dateFilter, datepickerConfig) {
			var self = this,
				ngModelCtrl = { $setViewValue: angular.noop }; // nullModelCtrl;

			// Modes chain
			this.modes = ['day', 'month', 'year'];

			// Configuration attributes
			angular.forEach(['formatDay', 'formatMonth', 'formatYear', 'formatDayHeader', 'formatDayTitle', 'formatMonthTitle',
				'minMode', 'maxMode', 'showWeeks', 'startingDay', 'yearRange'], function( key, index ) {
				self[key] = angular.isDefined($attrs[key]) ? (index < 8 ? $interpolate($attrs[key])($scope.$parent) : $scope.$parent.$eval($attrs[key])) : datepickerConfig[key];
			});

			// Watchable attributes
			angular.forEach(['minDate', 'maxDate'], function( key ) {
				if ( $attrs[key] ) {
					$scope.$parent.$watch($parse($attrs[key]), function(value) {
						self[key] = value ? new Date(value) : null;
						self.refreshView();
					});
				} else {
					self[key] = datepickerConfig[key] ? new Date(datepickerConfig[key]) : null;
				}
			});

			$scope.datepickerMode = $scope.datepickerMode || datepickerConfig.datepickerMode;
			$scope.uniqueId = 'datepicker-' + $scope.$id + '-' + Math.floor(Math.random() * 10000);
			$scope.fieldId = $attrs.fieldId ? $attrs.fieldId + '-today' : '';
			this.activeDate = angular.isDefined($attrs.initDate) ? $scope.$parent.$eval($attrs.initDate) : new Date();

			this.init = function( ngModelCtrl_ ) {
				ngModelCtrl = ngModelCtrl_;
				ngModelCtrl.$render = function() {
					self.render();
				};

				if (angular.isString(ngModelCtrl.$modelValue)) {
					ngModelCtrl.$render();
				}
			};

			this.render = function() {
				if ( ngModelCtrl.$modelValue ) {
					var date = new Date( ngModelCtrl.$modelValue ),
						isValid = !isNaN(date);

					if ( isValid ) {
						this.activeDate = date;
					} else {
						$log.error('Datepicker directive: "ng-model" value must be a Date object, a number of milliseconds since 01.01.1970 or a string representing an RFC2822 or ISO 8601 date.');
					}
					ngModelCtrl.$setValidity('date', isValid);
				}
				this.refreshView();
			};

			this.refreshView = function() {
				if ( this.element ) {
					this._refreshView();

					var date = ngModelCtrl.$modelValue ? new Date(ngModelCtrl.$modelValue) : null;
					ngModelCtrl.$setValidity('date-disabled', !date || (this.element && !this.isDisabled(date)));
					$scope.todayDisabled = this.isDisabled(new Date());
				}
			};

			this.isSelected = function(date) {
				var model = ngModelCtrl.$modelValue ? new Date(ngModelCtrl.$modelValue) : null;

				if (typeof this.compare === 'function') {
					return model && this.compare(date, model) === 0;
				}

				return false;
			};

			this.createDateObject = function(date, format) {
				return {
					date: date,
					label: dateFilter(date, format),
					selected: this.isSelected(date),
					disabled: this.isDisabled(date),
					current: this.compare(date, new Date()) === 0
				};
			};

			this.isDisabled = function( date ) {
				return ((this.minDate && this.compare(date, this.minDate) < 0) || (this.maxDate && this.compare(date, this.maxDate) > 0) || ($attrs.dateDisabled && $scope.dateDisabled({date: date, mode: $scope.datepickerMode})));
			};

			// Split array into smaller arrays
			this.split = function(arr, size) {
				var arrays = [];
				while (arr.length > 0) {
					arrays.push(arr.splice(0, size));
				}
				return arrays;
			};

			$scope.select = function( date ) {
				if ( $scope.datepickerMode === self.minMode ) {
					var dt = ngModelCtrl.$modelValue ? new Date( ngModelCtrl.$modelValue ) : new Date(0, 0, 0, 0, 0, 0, 0);
					dt.setFullYear( date.getFullYear(), date.getMonth(), date.getDate() );
					
					ngModelCtrl.$setViewValue( dt );
					ngModelCtrl.$render();
				} else {
					self.activeDate = date;
					$scope.datepickerMode = self.modes[ self.modes.indexOf( $scope.datepickerMode ) - 1 ];
					focusElement();
				}
			};

			$scope.selectToday = function() {
				$scope.datepickerMode = self.minMode;
				$scope.select(new Date());
			};

			$scope.isTodaySelected = function () {
				return self.isSelected(new Date());
			};

			$scope.move = function( direction ) {
				var year = self.activeDate.getFullYear() + direction * (self.step.years || 0),
					month = self.activeDate.getMonth() + direction * (self.step.months || 0);
				self.activeDate.setFullYear(year, month, 1);
				self.refreshView();
			};

			$scope.toggleMode = function( direction ) {
				direction = direction || 1;

				if (($scope.datepickerMode === self.maxMode && direction === 1) || ($scope.datepickerMode === self.minMode && direction === -1)) {
					return;
				}

				$scope.datepickerMode = self.modes[ self.modes.indexOf( $scope.datepickerMode ) + direction ];
				focusElement();
			};

			// Key event mapper
			$scope.keys = { 13:'enter', 32:'space', 33:'pageup', 34:'pagedown', 35:'end', 36:'home', 37:'left', 38:'up', 39:'right', 40:'down' };

			var focusEvent;
			var focusElement = function(event) {
				focusEvent = event || focusEvent;
				$timeout(function() {
					self.element.parent().focus();
				}, 0 , false);
			};

			// Listen for focus requests from popup directive
			$scope.$on('datepicker.focus', focusElement);

			$scope.keydown = function( evt ) {
				var key = $scope.keys[evt.which];

				if ( !key || evt.shiftKey || evt.altKey ) {
					return;
				}

				evt.preventDefault();
				evt.stopPropagation();

				if (key === 'enter' || key === 'space') {
					if ( self.isDisabled(self.activeDate)) {
						return; // do nothing
					}
					$scope.select(self.activeDate);
					focusElement();
				} else if (evt.ctrlKey && (key === 'up' || key === 'down')) {
					$scope.toggleMode(key === 'up' ? 1 : -1);
					focusElement();
				} else {
					self.handleKeyDown(key, evt);
					self.refreshView();
				}
			};
		})

		.directive( 'calendarDatePicker', function () {
			return {
				restrict: 'EA',
				replace: true,
				templateUrl: 'src/ui-components/form/controls/simple/calendar-date-picker/datepicker_template.html',
				scope: {
					datepickerMode: '=?',
					dateDisabled: '&'
				},
				require: ['calendarDatePicker', '?^ngModel'],
				controller: 'DatepickerController',
				link: function(scope, element, attrs, ctrls) {
					var datepickerCtrl = ctrls[0], ngModelCtrl = ctrls[1];

					// see: https://github.com/angular-ui/bootstrap/issues/2499
					scope.datepickerMode = datepickerCtrl.minMode;

					if ( ngModelCtrl ) {
						datepickerCtrl.init( ngModelCtrl );
					}
				}
			};
		})

		.config(function ($provide) {
			$provide.decorator('daypickerDirective', function($delegate) {
				var directive = $delegate[0];
				directive.require = '^calendarDatePicker';
				return $delegate;
			});

			$provide.decorator('monthpickerDirective', function($delegate) {
				var directive = $delegate[0];
				directive.require = '^calendarDatePicker';
				return $delegate;
			});

			$provide.decorator('yearpickerDirective', function($delegate) {
				var directive = $delegate[0];
				directive.require = '^calendarDatePicker';
				return $delegate;
			});
		});
});
